// Copyright 2014 Google Inc. All Rights Reserved.

#include "InputSource.h"

void InputSource::addDiscoveryInfo(ServiceDiscoveryResponse* sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    InputSourceService* iss = service->mutable_input_source_service();

    for (set<int32_t>::iterator it = mKeycodes.begin(); it != mKeycodes.end(); ++it) {
        iss->add_keycodes_supported(*it);
    }

    if (mHasTouchScreen) {
        InputSourceService::TouchScreen* ts = iss->add_touchscreen();
        ts->set_width(mTouchScreenWidth);
        ts->set_height(mTouchScreenHeight);
        ts->set_type(mTouchScreenType);
    }

    if (mHasTouchPad) {
        InputSourceService::TouchPad* tp = iss->add_touchpad();
        tp->set_width(mTouchPadWidth);
        tp->set_height(mTouchPadHeight);
        tp->set_physical_width(mTouchPadPhysWidth);
        tp->set_physical_height(mTouchPadPhysHeight);
        tp->set_ui_navigation(mTouchPadUiNavigation);
    }

    for (set<FeedbackEvent>::iterator it = mFeedbackEvents.begin();
            it != mFeedbackEvents.end(); ++it) {
        iss->add_feedback_events_supported(*it);
    }
}

int InputSource::routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*)msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);

    switch (type) {
        case INPUT_MESSAGE_KEY_BINDING_REQUEST: {
            KeyBindingRequest req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleKeyBindingRequest(req);
            }
            break;
        }
        case INPUT_MESSAGE_INPUT_FEEDBACK: {
            InputFeedback evt;
            if (PARSE_PROTO(evt, ptr, len)) {
                ret = handleInputFeedback(evt);
            }
            break;
        }
    }
    return ret;
}

bool InputSource::isBound(int32_t keycode) {
    return mBoundKeycodes.find(keycode) != mBoundKeycodes.end();
}

int InputSource::handleKeyBindingRequest(const KeyBindingRequest& req) {
    int status = STATUS_SUCCESS;
    for (int i = 0; i < req.keycodes_size(); i++) {
        if (mKeycodes.find(req.keycodes(i)) == mKeycodes.end()) {
            status = STATUS_KEYCODE_NOT_BOUND;
        }
    }

    if (status == STATUS_SUCCESS) {
        mBoundKeycodes.clear();
        for (int i = 0; i < req.keycodes_size(); i++) {
            mBoundKeycodes.insert(req.keycodes(i));
        }
    }

    KeyBindingResponse resp;
    resp.set_status(status);
    IoBuffer buf;
    mRouter->marshallProto(INPUT_MESSAGE_KEY_BINDING_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}

int InputSource::handleInputFeedback(const InputFeedback& feedback) {
    if (mFeedbackEvents.find(feedback.event()) != mFeedbackEvents.end()) {
        mCallbacks->onInputFeedback(feedback);
        return STATUS_SUCCESS;
    }
    return STATUS_UNEXPECTED_MESSAGE;
}

int InputSource::sendKey(uint64_t timestamp, uint32_t keycode, bool isDown,
            bool longPress, uint32_t metaState) {
    if (!isBound(keycode)) {
        return STATUS_KEYCODE_NOT_BOUND;
    }
    InputReport report;
    KeyEvent* event = report.mutable_key_event();
    KeyEvent_Key* key = event->add_keys();
    key->set_keycode(keycode);
    key->set_down(isDown);
    key->set_metastate(metaState);
    key->set_longpress(longPress);
    sendInputReport(timestamp, &report);
    return STATUS_SUCCESS;
}

int InputSource::reportKey(uint64_t timestamp, uint32_t keycode, bool isDown, uint32_t metaState) {
    return sendKey(timestamp, keycode, isDown, false, metaState);
}

int InputSource::reportKeyLongPress(uint64_t timestamp, uint32_t keycode, uint32_t metaState) {
    return sendKey(timestamp, keycode, true, true, metaState);
}

int InputSource::reportAbsolute(uint64_t timestamp, uint32_t keycode, int32_t value) {
    if (!isBound(keycode)) {
        return STATUS_KEYCODE_NOT_BOUND;
    }
    InputReport report;
    AbsoluteEvent* event = report.mutable_absolute_event();
    AbsoluteEvent_Abs* abs = event->add_data();
    abs->set_keycode(keycode);
    abs->set_value(value);
    sendInputReport(timestamp, &report);
    return STATUS_SUCCESS;
}

int InputSource::reportRelative(uint64_t timestamp, uint32_t keycode, int32_t delta) {
    if (!isBound(keycode)) {
        return STATUS_KEYCODE_NOT_BOUND;
    }
    InputReport report;
    RelativeEvent* event = report.mutable_relative_event();
    RelativeEvent_Rel* rel = event->add_data();
    rel->set_keycode(keycode);
    rel->set_delta(delta);
    sendInputReport(timestamp, &report);
    return STATUS_SUCCESS;
}

void InputSource::populateTouchEvent(TouchEvent* event, uint64_t timestamp, int32_t numPointers,
        const uint32_t* pointerIds, const uint32_t* x, const uint32_t* y, int action,
        int actionIndex) {
    for (int i = 0; i < numPointers; i++) {
        TouchEvent_Pointer* ptr = event->add_pointer_data();
        ptr->set_pointer_id(pointerIds[i]);
        ptr->set_x(x[i]);
        ptr->set_y(y[i]);
    }
    event->set_action_index(actionIndex);
    event->set_action((PointerAction) action);
}

int InputSource::reportTouch(uint64_t timestamp, int32_t numPointers, const uint32_t* pointerIds,
        const uint32_t* x, const uint32_t* y, int action, int actionIndex) {
    if (!mHasTouchScreen) {
        return STATUS_INVALID_INPUT;
    }
    InputReport report;
    TouchEvent* event = report.mutable_touch_event();
    populateTouchEvent(event, timestamp, numPointers, pointerIds, x, y, action, actionIndex);
    sendInputReport(timestamp, &report);
    return STATUS_SUCCESS;
}

int InputSource::reportTouchPad(uint64_t timestamp, int32_t numPointers, const uint32_t* pointerIds,
            const uint32_t* x, const uint32_t* y, int action, int actionIndex) {
    if (!mHasTouchPad) {
        return STATUS_INVALID_INPUT;
    }
    InputReport report;
    TouchEvent* event = report.mutable_touchpad_event();
    populateTouchEvent(event, timestamp, numPointers, pointerIds, x, y, action, actionIndex);
    sendInputReport(timestamp, &report);
    return STATUS_SUCCESS;
}

void InputSource::sendInputReport(uint64_t timestamp, InputReport* report) {
    report->set_timestamp(timestamp);
    IoBuffer buf;
    mRouter->marshallProto(INPUT_MESSAGE_INPUT_REPORT, *report, &buf);
    queueOutgoing(buf.raw(), buf.size());
}
